/*
* sstar_usb_phy.c- Sigmastar
*
* Copyright (C) 2021 Sigmastar Technology Corp.
* All rights reserved.
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
*/

#include <linux/init.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>

MODULE_DESCRIPTION("USB DP DM STATUS");
MODULE_LICENSE("GPL v2");

static int sstar_utmi_write(u32 val, u32 reg)
{
    *(unsigned short*)(0xfd284200 + (reg << 2)) = val;
    return 0;
}

static int sstar_utmi_read(u32 reg)
{
    return (int)(*(unsigned short*)(0xfd284200 + (reg << 2)));
}

void sstar_utmi_deinit(void)
{
    // bit0: RX sw reset; bit1: Tx sw reset; bit8: Tx FSM sw reset;
    // bit12: pwr good reset
    sstar_utmi_write(sstar_utmi_read(0x03) | 0x0103, 0x03);
    sstar_utmi_write(sstar_utmi_read(0x08) | 0x1000, 0x08);
    mdelay(1);

    // clear reset
    // bit0: RX sw reset; bit1: Tx sw reset; bit8: Tx FSM sw reset;
    // bit12: pwr good reset
    sstar_utmi_write(sstar_utmi_read(0x03) & (~0x0103), 0x03);
    sstar_utmi_write(sstar_utmi_read(0x08) & (~0x1000), 0x08);

    // power down utmi
    sstar_utmi_write(0x7F03, 0x0);
    mdelay(5);
    printk("<USB>[GADGET] Deinit phy\n");
}

static void sstar_utmi_init(void)
{
    sstar_utmi_write(0x0C2F, 0x04);

    sstar_utmi_write(0x6BC3, 0x0); //Turn on UPLL, reg_pdn: bit<9> reg_pdn: bit<15>, bit <2> ref_pdn
    mdelay(1);
    sstar_utmi_write(0x0069, 0x0); //Turn on UPLL, reg_pdn: bit<9>
    mdelay(2);
    sstar_utmi_write(0x0001, 0x0); //Turn all (including hs_current) use override mode
    mdelay(3);

    // Turn on UTMI if it was powered down
    if (0x0001 != sstar_utmi_read(0))
    {
        sstar_utmi_write(0x0001, 0x0); //Turn all (including hs_current) use override mode
        mdelay(3);
    }

    sstar_utmi_write((sstar_utmi_read(0x1e) | 0x0001), 0x3c); // set CA_START as 1
    mdelay(10);
    sstar_utmi_write((sstar_utmi_read(0x1e) & ~0x0001), 0x3c);// release CA_START

    while (0 == (sstar_utmi_read(0x1e) & 0x0002)); // polling bit <1> (CA_END)

    sstar_utmi_write(((sstar_utmi_read(0x03) & 0xFF9F) | 0x0040), 0x03);    //reg_tx_force_hs_current_enable
    sstar_utmi_write((sstar_utmi_read(0x01) | 0x2800), 0x01);               //Disconnect window select
    sstar_utmi_write((sstar_utmi_read(0x01) & 0xEFFF), 0x01);               //Disconnect window select
    sstar_utmi_write((sstar_utmi_read(0x03) & 0xFDFF), 0x03);               //Disable improved CDR

    sstar_utmi_write((sstar_utmi_read(0x04) | 0x8100), 0x04);               // UTMI RX anti-dead-loc, ISI effect improvement
    sstar_utmi_write((sstar_utmi_read(0x0a) | 0x2000), 0xa);               // Chirp signal source select
    sstar_utmi_write((sstar_utmi_read(0x05) | 0x8000), 0x05);               // set reg_ck_inv_reserved[6] to solve timing problem

    sstar_utmi_write((sstar_utmi_read(0x16) & ~0x000C), 0x16);
    sstar_utmi_write((sstar_utmi_read(0x16) | 0x0010), 0x16);
    sstar_utmi_write((sstar_utmi_read(0x16) | 0x0200), 0x16);

    sstar_utmi_write((sstar_utmi_read(0x01) | 0x0080), 0x02);               //avoid glitch
    sstar_utmi_write((sstar_utmi_read(0x0) | 0x0002), 0x0);
}

static void sstar_usb_dpdm_pull_up(int enable)
{
    if (enable)
        sstar_utmi_write((sstar_utmi_read(0x0) | 0x0018), 0x0);
    else
        sstar_utmi_write((sstar_utmi_read(0x0) & ~0x0018), 0x0);
}

unsigned char u8SolarType = 0;

static ssize_t sstar_utmi_dp_dm_status(struct class *class, struct class_attribute *attr, char *buf)
{
    return sprintf(buf, "%X\n", u8SolarType);
}

static struct class_attribute usb_phy_class_attrs[] = {
    __ATTR(solar_type, S_IRUGO, sstar_utmi_dp_dm_status, NULL),
    __ATTR_NULL
};

struct class usb_phy_class = {
    .name           = "usb_phy",
    .owner          = THIS_MODULE,
    .class_attrs    = usb_phy_class_attrs,
};

static int usb_phy_init(void)
{
    printk(KERN_INFO "\nUSB DpDm init!\n");

    sstar_utmi_init();
    sstar_usb_dpdm_pull_up(1);  //DP DM pull up

    class_register(&usb_phy_class);

    u8SolarType = (sstar_utmi_read(0x18) >> 12) & 3;

    sstar_usb_dpdm_pull_up(0);

    return 0;
}

static void usb_phy_exit(void)
{
    sstar_usb_dpdm_pull_up(0);
    sstar_utmi_deinit();

    printk(KERN_INFO "USB DpDm reset!\n");
}

module_init(usb_phy_init);
module_exit(usb_phy_exit);
